"use client";

import React, { useEffect, useState } from "react";
import InnerBanner from "../../../components/InnerBanner";
import "../projects.css";
import Link from "next/link";
import { useCart } from "@/context/CartContext";

function stripHtml(html: string): string {
    if (typeof window !== "undefined") {
        const div = document.createElement("div");
        div.innerHTML = html;
        return div.textContent || div.innerText || "";
    }
    return html.replace(/<[^>]*>?/gm, "").trim();
}

type SelectionState = {
    type?: string;
    medium?: string;
    session?: string;
};

interface SubjectDetailsClientProps {
    courseData: any;
    projectsList: any[];
    courseCode: string;
}

export default function SubjectDetailsClient({ courseData, projectsList, courseCode }: SubjectDetailsClientProps) {
    const [mounted, setMounted] = useState(false);
    const [search, setSearch] = useState("");

    useEffect(() => {
        if (!document.querySelector('link[href*="bootstrap.min.css"]')) {
            const bs = document.createElement("link");
            bs.rel = "stylesheet";
            bs.href = "https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css";
            document.head.appendChild(bs);
        }
        if (!document.querySelector('link[href*="font-awesome"]')) {
            const fa = document.createElement("link");
            fa.rel = "stylesheet";
            fa.href = "https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css";
            document.head.appendChild(fa);
        }
        setTimeout(() => setMounted(true), 100);
    }, []);

    const { addToCart } = useCart();
    const [selections, setSelections] = useState<Record<string, SelectionState>>({});
    const [activeCartSlug, setActiveCartSlug] = useState<string | null>(null);

    const headline = courseData?.course_headline_projects || "";
    const breadcrumbHeadline = courseData?.breadcrumb_headline_projects || "";
    const description = courseData?.description_projects ? stripHtml(courseData.description_projects) : "";

    const updateSelection = (
        slug: string,
        field: keyof SelectionState,
        value: string
    ) => {
        setSelections(prev => ({
            ...prev,
            [slug]: {
                ...prev[slug],
                [field]: value,
            },
        }));
    };

    const handleAddToCart = async (item: any) => {
        const selection = selections[item.project_slug];
        if (!selection) return;

        setActiveCartSlug(item.project_slug);

        const rawString = `${courseCode}|${item.project_slug}|${Date.now()}`;

        try {
            await addToCart({
                id: `project-${rawString}`,
                name: item.title ?? "",
                price: item.sale ?? 0,
                qty: 1,
                img: item.img ?? "",
                courseCode,
                itemCode: item.project_slug,
                subjectIds: null,
                selectedType: null,
                selectedMedium: selection.medium!,
                selectedSession: null,
                dataSaveId: null,
                pageName: "project-page",
            });
        } finally {
            setActiveCartSlug(null);
        }
    };

    const filteredProjects = projectsList.filter(p =>
        p.code?.toLowerCase().includes(search.toLowerCase())
    );

    if (!mounted) return null;

    return (
        <>
            {/* Inner Banner */}
            <InnerBanner title={breadcrumbHeadline} breadcrumb={breadcrumbHeadline} />

            {/* Introduction Heading */}
            <div className="container mt-5">
                <div className="d-lg-flex d-md-block d-block align-items-center gap-2 mb-4 justify-content-between">
                    <div className='d-flex '>
                        <h2 className="fw-bold">
                            {headline}
                        </h2>
                        {/* <p dangerouslySetInnerHTML={{ __html: description }} /> */}
                    </div>

                    {/* 🔥 Search Box */}
                    <div className="search-box mb-4 w-50 position-relative">
                        <input
                            type="text"
                            className="form-control "
                            placeholder="Search by Project Code..."
                            value={search}
                            onChange={(e) => setSearch(e.target.value)}
                        />

                        <i className="fa-solid fa-magnifying-glass"></i>
                    </div>
                </div>
            </div>

            <div className="project-listing-sec py-5">
                <div className="container">
                    <div className="row g-4">
                        {filteredProjects.length > 0 ? (
                            filteredProjects.map((item: any, i: number) => {
                                const discount = Math.round(((item.price - item.sale) / item.price) * 100);

                                const currentSelection = selections[item.project_slug];
                                const isDisabled =
                                    !currentSelection?.medium ||
                                    (activeCartSlug === item.project_slug);

                                return (
                                    <div key={i} className="col-12 col-md-6 col-lg-3">
                                        <div className="product-card h-100 d-flex flex-column">
                                            <div className="product-image-wrapper">
                                                <img
                                                    src={item.img}
                                                    className="product-image"
                                                    alt={item.title}
                                                />
                                            </div>

                                            <div className="product-body text-start p-3 flex-grow-1">
                                                <h5 className="card-code mb-2">{item.code}</h5>
                                                <h6 className="card-title-sm text-muted mb-3">{item.title}</h6>

                                                {/* Languages */}
                                                <select
                                                    className="form-select form-select-sm mb-2 shadow-none border-secondary-subtle"
                                                    value={currentSelection?.medium || ""}
                                                    onChange={e =>
                                                        updateSelection(item.project_slug, "medium", e.target.value)
                                                    }
                                                    aria-label="Language Selection"
                                                >
                                                    <option value="">-- Select Medium --</option>
                                                    {item.medium?.map((language: { enc_id: string; language_name: string }) => (
                                                        <option key={language.enc_id} value={language.enc_id}>{language.language_name}</option>
                                                    ))}
                                                </select>

                                                <div className="d-flex align-items-center mb-3 mt-3">
                                                    <h4 className="mb-0 fw-bold">₹{item.sale}</h4>
                                                    <span className="text-muted ms-2 text-decoration-line-through small">Only</span>
                                                    <span className="text-muted ms-1 text-decoration-line-through small">MRP ₹{item.price}</span>
                                                    <span className="badge bg-danger ms-auto">{discount}% Off</span>
                                                </div>

                                                <small
                                                    className="text-danger mb-1 d-block"
                                                    style={{ fontSize: "0.72rem", fontStyle: "italic" }}
                                                >
                                                    ** Medium, and Session are mandatory fields.
                                                </small>

                                                <div className="gap-2">
                                                    {/* VIEW */}
                                                    <Link href={`/projects/${courseCode}/${item.project_slug}`} className="btn bg-dark border text-white mb-2 text-center btn-mb w-100 ">
                                                        View
                                                    </Link>

                                                    {/* ADD TO CART */}
                                                    <button
                                                        className="btn btn-primary py-2 rounded-3 btn-mb w-100 text-center"
                                                        disabled={isDisabled}
                                                        onClick={() => handleAddToCart(item)}
                                                    >
                                                        {activeCartSlug === item.project_slug ? (
                                                            <>
                                                                <span className="spinner-border spinner-border-sm" /> Adding...
                                                            </>
                                                        ) : (
                                                            "Add to Cart"
                                                        )}
                                                    </button>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                );
                            })
                        ) : (
                            <div className="col-12">
                                <div className="empty-state">
                                    <div className="empty-icon">
                                        <i className="fa-solid fa-folder-open"></i>
                                    </div>
                                    <h3 className="mb-3">No Projects Available</h3>
                                    <p className="text-muted">
                                        We're currently preparing projects for this category. Please check back soon!
                                    </p>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </>
    );
}
